<?php
require_once __DIR__ . '/../includes/functions.php';
requireAdmin();
$db = getDB();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'save_service') {
        $id       = (int)($_POST['id'] ?? 0);
        $catId    = (int)$_POST['category_id'];
        $name     = sanitize($_POST['name']);
        $desc     = sanitize($_POST['description'] ?? '');
        $price    = (float)$_POST['base_price'];
        $addon    = (float)($_POST['postpay_addon'] ?? 100);
        $hasPlan  = (int)($_POST['has_payment_plans'] ?? 0);
        $isTrain  = (int)($_POST['is_training'] ?? 0);
        $classes  = $isTrain ? (int)$_POST['classes_count'] : null;
        $ppc      = $isTrain ? (float)$_POST['price_per_class'] : null;
        $sort     = (int)($_POST['sort_order'] ?? 0);
        $active   = (int)($_POST['is_active'] ?? 1);

        if ($id) {
            $stmt = $db->prepare("UPDATE services SET category_id=?,name=?,description=?,base_price=?,postpay_addon=?,has_payment_plans=?,is_training=?,classes_count=?,price_per_class=?,sort_order=?,is_active=? WHERE id=?");
            $stmt->bind_param('issddiiiidii', $catId,$name,$desc,$price,$addon,$hasPlan,$isTrain,$classes,$ppc,$sort,$active,$id);
        } else {
            $stmt = $db->prepare("INSERT INTO services (category_id,name,description,base_price,postpay_addon,has_payment_plans,is_training,classes_count,price_per_class,sort_order,is_active) VALUES (?,?,?,?,?,?,?,?,?,?,?)");
            $stmt->bind_param('issddiiiidi', $catId,$name,$desc,$price,$addon,$hasPlan,$isTrain,$classes,$ppc,$sort,$active);
        }
        $stmt->execute();
        setFlash('success', 'Service saved successfully.');

    } elseif ($action === 'delete_service') {
        $id = (int)$_POST['id'];
        $db->query("DELETE FROM services WHERE id=$id");
        setFlash('success', 'Service deleted.');

    } elseif ($action === 'save_category') {
        $id   = (int)($_POST['id'] ?? 0);
        $name = sanitize($_POST['name']);
        $slug = strtolower(preg_replace('/[^a-z0-9]+/i', '-', $name));
        $sort = (int)($_POST['sort_order'] ?? 0);
        if ($id) {
            $db->prepare("UPDATE service_categories SET name=?,slug=?,sort_order=? WHERE id=?")->bind_param('ssii',$name,$slug,$sort,$id)->execute();
        } else {
            $db->prepare("INSERT INTO service_categories (name,slug,sort_order) VALUES (?,?,?)")->bind_param('ssi',$name,$slug,$sort)->execute();
        }
        setFlash('success', 'Category saved.');

    } elseif ($action === 'save_plan') {
        $id    = (int)($_POST['id'] ?? 0);
        $type  = in_array($_POST['plan_type'],['prepay','postpay']) ? $_POST['plan_type'] : 'prepay';
        $terms = sanitize($_POST['terms']);
        if ($id) {
            $db->prepare("UPDATE payment_plans SET plan_type=?,terms=? WHERE id=?")->bind_param('ssi',$type,$terms,$id)->execute();
        } else {
            $db->prepare("INSERT INTO payment_plans (plan_type,terms) VALUES (?,?)")->bind_param('ss',$type,$terms)->execute();
        }
        setFlash('success', 'Payment plan updated.');

    } elseif ($action === 'save_discount') {
        $id     = (int)($_POST['id'] ?? 0);
        $amount = (float)$_POST['amount'];
        $label  = sanitize($_POST['label'] ?? '');
        if ($id) {
            $db->prepare("UPDATE discount_options SET amount=?,label=? WHERE id=?")->bind_param('dsi',$amount,$label,$id)->execute();
        } else {
            $db->prepare("INSERT INTO discount_options (amount,label) VALUES (?,?)")->bind_param('ds',$amount,$label)->execute();
        }
        setFlash('success', 'Discount option saved.');

    } elseif ($action === 'delete_discount') {
        $id = (int)$_POST['id'];
        $db->query("DELETE FROM discount_options WHERE id=$id");
        setFlash('success', 'Discount deleted.');
    }

    header('Location: /admin/services.php');
    exit;
}

$categories = $db->query("SELECT * FROM service_categories ORDER BY sort_order")->fetch_all(MYSQLI_ASSOC);
$services   = $db->query("SELECT s.*, sc.name as cat_name FROM services s JOIN service_categories sc ON s.category_id=sc.id ORDER BY s.category_id, s.sort_order")->fetch_all(MYSQLI_ASSOC);
$payPlans   = $db->query("SELECT * FROM payment_plans ORDER BY plan_type")->fetch_all(MYSQLI_ASSOC);
$discounts  = $db->query("SELECT * FROM discount_options ORDER BY amount")->fetch_all(MYSQLI_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Services & Pricing — ProCRM Admin</title>
<link rel="stylesheet" href="/assets/css/style.css">
</head>
<body>
<div class="layout">
  <?php include __DIR__ . '/../includes/admin-sidebar.php'; ?>
  <div class="main-content">
    <div class="topbar">
      <button class="topbar-toggle" id="sidebarToggle">
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="3" y1="6" x2="21" y2="6"/><line x1="3" y1="12" x2="21" y2="12"/><line x1="3" y1="18" x2="21" y2="18"/></svg>
      </button>
      <span class="topbar-title">Services & Pricing</span>
    </div>
    <div class="page">
      <?= renderFlash() ?>

      <div class="page-header" style="display:flex;justify-content:space-between;align-items:center">
        <div>
          <h2>Services & Pricing</h2>
          <p>Manage service categories, pricing, and payment options.</p>
        </div>
        <button class="btn btn-primary" data-modal="modal-new-service">+ Add Service</button>
      </div>

      <!-- Services Table -->
      <div class="card" style="margin-bottom:20px">
        <div class="card-header">
          <div class="card-title">All Services</div>
        </div>
        <div class="table-wrap">
          <table>
            <thead>
              <tr><th>Service</th><th>Category</th><th>Base Price</th><th>PostPay</th><th>Type</th><th>Status</th><th>Actions</th></tr>
            </thead>
            <tbody>
              <?php foreach ($services as $s): ?>
              <tr>
                <td><div style="font-weight:600"><?= htmlspecialchars($s['name']) ?></div></td>
                <td><span class="badge badge-pending" style="background:rgba(255,255,255,0.08);color:var(--white)"><?= htmlspecialchars($s['cat_name']) ?></span></td>
                <td><strong>$<?= number_format($s['base_price'],2) ?></strong><?php if ($s['is_training']): ?> <span class="text-muted" style="font-size:12px">/class</span><?php endif; ?></td>
                <td><?php if (!$s['is_training']): ?>$<?= number_format($s['base_price']+$s['postpay_addon'],2) ?><?php else: ?>—<?php endif; ?></td>
                <td><span class="badge <?= $s['is_training'] ? 'badge-confirmed' : 'badge-prepay' ?>"><?= $s['is_training'] ? 'Training' : 'Campaign' ?></span></td>
                <td><span class="badge <?= $s['is_active'] ? 'badge-approved' : 'badge-rejected' ?>"><?= $s['is_active'] ? 'Active' : 'Inactive' ?></span></td>
                <td>
                  <div style="display:flex;gap:6px">
                    <button class="btn btn-outline btn-sm" onclick="editService(<?= htmlspecialchars(json_encode($s)) ?>)">Edit</button>
                    <form method="POST" style="display:inline" onsubmit="return confirm('Delete this service?')">
                      <input type="hidden" name="action" value="delete_service">
                      <input type="hidden" name="id" value="<?= $s['id'] ?>">
                      <button type="submit" class="btn btn-danger btn-sm">Del</button>
                    </form>
                  </div>
                </td>
              </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>

      <!-- Payment Plans -->
      <div style="display:grid;grid-template-columns:1fr 1fr;gap:20px">
        <div class="card">
          <div class="card-header">
            <div class="card-title">Payment Plans</div>
            <button class="btn btn-outline btn-sm" data-modal="modal-new-plan">+ Add</button>
          </div>
          <?php foreach ($payPlans as $pp): ?>
          <div style="background:var(--black);border:1px solid var(--border);border-radius:8px;padding:16px;margin-bottom:12px">
            <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:10px">
              <span class="badge badge-<?= $pp['plan_type'] ?>"><?= ucfirst($pp['plan_type']) ?></span>
              <button class="btn btn-outline btn-sm" onclick="editPlan(<?= htmlspecialchars(json_encode($pp)) ?>)">Edit</button>
            </div>
            <p style="font-size:13px;color:var(--text-muted);line-height:1.6"><?= htmlspecialchars($pp['terms']) ?></p>
          </div>
          <?php endforeach; ?>
        </div>

        <div class="card">
          <div class="card-header">
            <div class="card-title">Discount Options</div>
            <button class="btn btn-outline btn-sm" data-modal="modal-new-discount">+ Add</button>
          </div>
          <div style="display:flex;flex-direction:column;gap:10px">
            <?php foreach ($discounts as $d): ?>
            <div style="display:flex;align-items:center;justify-content:space-between;background:var(--black);border:1px solid var(--border);border-radius:8px;padding:12px 16px">
              <div>
                <div style="font-weight:700;font-size:18px">$<?= number_format($d['amount'],0) ?></div>
                <div style="font-size:12px;color:var(--text-muted)"><?= htmlspecialchars($d['label']) ?></div>
              </div>
              <div style="display:flex;gap:6px">
                <button class="btn btn-outline btn-sm" onclick="editDiscount(<?= htmlspecialchars(json_encode($d)) ?>)">Edit</button>
                <form method="POST" onsubmit="return confirm('Delete?')" style="display:inline">
                  <input type="hidden" name="action" value="delete_discount">
                  <input type="hidden" name="id" value="<?= $d['id'] ?>">
                  <button type="submit" class="btn btn-danger btn-sm">Del</button>
                </form>
              </div>
            </div>
            <?php endforeach; ?>
          </div>
        </div>
      </div>

      <!-- Categories -->
      <div class="card" style="margin-top:20px">
        <div class="card-header">
          <div class="card-title">Service Categories</div>
          <button class="btn btn-outline btn-sm" data-modal="modal-new-category">+ Add</button>
        </div>
        <div style="display:flex;gap:12px;flex-wrap:wrap">
          <?php foreach ($categories as $cat): ?>
          <div style="background:var(--black);border:1px solid var(--border);border-radius:8px;padding:14px 18px;display:flex;align-items:center;gap:12px">
            <div>
              <div style="font-weight:600"><?= htmlspecialchars($cat['name']) ?></div>
              <div style="font-size:11px;color:var(--text-muted);font-family:var(--font-mono)"><?= $cat['slug'] ?></div>
            </div>
            <button class="btn btn-outline btn-sm" onclick="editCategory(<?= htmlspecialchars(json_encode($cat)) ?>)">Edit</button>
          </div>
          <?php endforeach; ?>
        </div>
      </div>

    </div>
  </div>
</div>

<!-- Modal: Add/Edit Service -->
<div class="modal-backdrop" id="modal-new-service" style="display:none">
  <div class="modal" style="max-width:600px">
    <div class="modal-header">
      <div class="modal-title" id="serviceModalTitle">Add Service</div>
      <button class="modal-close" data-dismiss="modal">✕</button>
    </div>
    <div class="modal-body">
      <form method="POST" id="serviceForm">
        <input type="hidden" name="action" value="save_service">
        <input type="hidden" name="id" id="svcId" value="">
        <div class="form-row">
          <div class="form-group">
            <label class="form-label">Category *</label>
            <select name="category_id" id="svcCat" class="form-control" required>
              <?php foreach ($categories as $cat): ?>
              <option value="<?= $cat['id'] ?>"><?= htmlspecialchars($cat['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="form-group">
            <label class="form-label">Service Name *</label>
            <input type="text" name="name" id="svcName" class="form-control" required>
          </div>
        </div>
        <div class="form-group">
          <label class="form-label">Description</label>
          <textarea name="description" id="svcDesc" class="form-control"></textarea>
        </div>
        <div class="form-row">
          <div class="form-group">
            <label class="form-label">Base Price ($) *</label>
            <input type="number" step="0.01" name="base_price" id="svcPrice" class="form-control" required>
          </div>
          <div class="form-group" id="postpayAddonGroup">
            <label class="form-label">PostPay Addon ($)</label>
            <input type="number" step="0.01" name="postpay_addon" id="svcAddon" class="form-control" value="100">
          </div>
        </div>
        <div class="form-row">
          <div class="form-group">
            <label class="form-label">Type</label>
            <select name="is_training" id="svcTraining" class="form-control" onchange="toggleTraining(this.value)">
              <option value="0">Campaign</option>
              <option value="1">Training</option>
            </select>
          </div>
          <div class="form-group">
            <label class="form-label">Sort Order</label>
            <input type="number" name="sort_order" id="svcSort" class="form-control" value="0">
          </div>
        </div>
        <div id="trainingFields" style="display:none">
          <div class="form-row">
            <div class="form-group">
              <label class="form-label">Number of Classes</label>
              <input type="number" name="classes_count" id="svcClasses" class="form-control">
            </div>
            <div class="form-group">
              <label class="form-label">Price Per Class ($)</label>
              <input type="number" step="0.01" name="price_per_class" id="svcPPC" class="form-control">
            </div>
          </div>
        </div>
        <div class="form-row">
          <div class="form-group">
            <label class="form-label">Has Payment Plans</label>
            <select name="has_payment_plans" id="svcPlans" class="form-control">
              <option value="1">Yes</option>
              <option value="0">No</option>
            </select>
          </div>
          <div class="form-group">
            <label class="form-label">Status</label>
            <select name="is_active" id="svcActive" class="form-control">
              <option value="1">Active</option>
              <option value="0">Inactive</option>
            </select>
          </div>
        </div>
        <button type="submit" class="btn btn-primary btn-full">Save Service</button>
      </form>
    </div>
  </div>
</div>

<!-- Modal: Payment Plan -->
<div class="modal-backdrop" id="modal-new-plan" style="display:none">
  <div class="modal">
    <div class="modal-header">
      <div class="modal-title">Payment Plan</div>
      <button class="modal-close" data-dismiss="modal">✕</button>
    </div>
    <div class="modal-body">
      <form method="POST">
        <input type="hidden" name="action" value="save_plan">
        <input type="hidden" name="id" id="planId" value="">
        <div class="form-group">
          <label class="form-label">Plan Type</label>
          <select name="plan_type" id="planType" class="form-control">
            <option value="prepay">Prepay</option>
            <option value="postpay">Postpay</option>
          </select>
        </div>
        <div class="form-group">
          <label class="form-label">Terms & Conditions</label>
          <textarea name="terms" id="planTerms" class="form-control" rows="5" required></textarea>
        </div>
        <button type="submit" class="btn btn-primary btn-full">Save Plan</button>
      </form>
    </div>
  </div>
</div>

<!-- Modal: Discount -->
<div class="modal-backdrop" id="modal-new-discount" style="display:none">
  <div class="modal">
    <div class="modal-header">
      <div class="modal-title">Discount Option</div>
      <button class="modal-close" data-dismiss="modal">✕</button>
    </div>
    <div class="modal-body">
      <form method="POST">
        <input type="hidden" name="action" value="save_discount">
        <input type="hidden" name="id" id="discountId" value="">
        <div class="form-row">
          <div class="form-group">
            <label class="form-label">Amount ($)</label>
            <input type="number" step="0.01" name="amount" id="discountAmt" class="form-control" required>
          </div>
          <div class="form-group">
            <label class="form-label">Label</label>
            <input type="text" name="label" id="discountLabel" class="form-control" placeholder="e.g. $10 Discount">
          </div>
        </div>
        <button type="submit" class="btn btn-primary btn-full">Save</button>
      </form>
    </div>
  </div>
</div>

<!-- Modal: Category -->
<div class="modal-backdrop" id="modal-new-category" style="display:none">
  <div class="modal">
    <div class="modal-header">
      <div class="modal-title">Service Category</div>
      <button class="modal-close" data-dismiss="modal">✕</button>
    </div>
    <div class="modal-body">
      <form method="POST">
        <input type="hidden" name="action" value="save_category">
        <input type="hidden" name="id" id="catId" value="">
        <div class="form-row">
          <div class="form-group">
            <label class="form-label">Category Name</label>
            <input type="text" name="name" id="catName" class="form-control" required>
          </div>
          <div class="form-group">
            <label class="form-label">Sort Order</label>
            <input type="number" name="sort_order" id="catSort" class="form-control" value="0">
          </div>
        </div>
        <button type="submit" class="btn btn-primary btn-full">Save Category</button>
      </form>
    </div>
  </div>
</div>

<script src="/assets/js/app.js"></script>
<script>
function toggleTraining(val) {
  document.getElementById('trainingFields').style.display  = val == '1' ? 'block' : 'none';
  document.getElementById('postpayAddonGroup').style.display = val == '1' ? 'none' : 'block';
}

function editService(s) {
  document.getElementById('serviceModalTitle').textContent = 'Edit Service';
  document.getElementById('svcId').value     = s.id;
  document.getElementById('svcCat').value    = s.category_id;
  document.getElementById('svcName').value   = s.name;
  document.getElementById('svcDesc').value   = s.description || '';
  document.getElementById('svcPrice').value  = s.base_price;
  document.getElementById('svcAddon').value  = s.postpay_addon;
  document.getElementById('svcTraining').value = s.is_training;
  document.getElementById('svcSort').value   = s.sort_order;
  document.getElementById('svcClasses').value = s.classes_count || '';
  document.getElementById('svcPPC').value    = s.price_per_class || '';
  document.getElementById('svcPlans').value  = s.has_payment_plans;
  document.getElementById('svcActive').value = s.is_active;
  toggleTraining(s.is_training);
  document.getElementById('modal-new-service').style.display = 'flex';
}

function editPlan(p) {
  document.getElementById('planId').value    = p.id;
  document.getElementById('planType').value  = p.plan_type;
  document.getElementById('planTerms').value = p.terms;
  document.getElementById('modal-new-plan').style.display = 'flex';
}

function editDiscount(d) {
  document.getElementById('discountId').value    = d.id;
  document.getElementById('discountAmt').value   = d.amount;
  document.getElementById('discountLabel').value = d.label;
  document.getElementById('modal-new-discount').style.display = 'flex';
}

function editCategory(c) {
  document.getElementById('catId').value   = c.id;
  document.getElementById('catName').value = c.name;
  document.getElementById('catSort').value = c.sort_order;
  document.getElementById('modal-new-category').style.display = 'flex';
}
</script>
</body>
</html>
