<?php
require_once __DIR__ . '/../config/database.php';

session_start();

// ─── Auth Helpers ────────────────────────────────────────────────────────────
function isLoggedIn() {
    return isset($_SESSION['user_id']) && $_SESSION['role'] === 'client';
}

function isAdminLoggedIn() {
    return isset($_SESSION['admin_id']) && $_SESSION['role'] === 'admin';
}

function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: /client/login.php');
        exit;
    }
}

function requireAdmin() {
    if (!isAdminLoggedIn()) {
        header('Location: /admin/login.php');
        exit;
    }
}

// ─── Security ────────────────────────────────────────────────────────────────
function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)));
}

function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

function generateCode($length = 6) {
    return str_pad(random_int(0, 999999), $length, '0', STR_PAD_LEFT);
}

// ─── Email ───────────────────────────────────────────────────────────────────
function sendEmail($to, $subject, $htmlBody, $plainBody = '') {
    $db = getDB();
    
    // Log it
    $stmt = $db->prepare("INSERT INTO email_logs (to_email, subject, body) VALUES (?,?,?)");
    $stmt->bind_param('sss', $to, $subject, $htmlBody);
    $stmt->execute();

    // Try PHPMailer if available, fallback to mail()
    if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
        $settings = getSettings();
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        try {
            $mail->isSMTP();
            $mail->Host = $settings['smtp_host'] ?? 'localhost';
            $mail->SMTPAuth = true;
            $mail->Username = $settings['smtp_user'] ?? '';
            $mail->Password = $settings['smtp_pass'] ?? '';
            $mail->SMTPSecure = 'tls';
            $mail->Port = $settings['smtp_port'] ?? 587;
            $mail->setFrom($settings['smtp_from'] ?? 'noreply@crm.com', $settings['site_name'] ?? 'CRM');
            $mail->addAddress($to);
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $htmlBody;
            $mail->AltBody = $plainBody ?: strip_tags($htmlBody);
            return $mail->send();
        } catch (Exception $e) {
            return false;
        }
    } else {
        // Basic PHP mail
        $headers  = "MIME-Version: 1.0\r\n";
        $headers .= "Content-type: text/html; charset=UTF-8\r\n";
        $headers .= "From: noreply@crm.com\r\n";
        return mail($to, $subject, $htmlBody, $headers);
    }
}

function emailTemplate($title, $content) {
    return "<!DOCTYPE html>
<html>
<head><meta charset='UTF-8'><style>
body{font-family:'Segoe UI',sans-serif;background:#0a0a0a;color:#fff;margin:0;padding:20px}
.container{max-width:600px;margin:0 auto;background:#111;border:1px solid #333;border-radius:8px;overflow:hidden}
.header{background:#000;padding:30px;text-align:center;border-bottom:2px solid #fff}
.header h1{margin:0;font-size:24px;font-weight:900;letter-spacing:3px;text-transform:uppercase}
.body{padding:40px}
.body h2{font-size:20px;margin-bottom:20px}
.code-box{background:#1a1a1a;border:1px solid #444;border-radius:6px;padding:20px;text-align:center;margin:20px 0}
.code{font-size:36px;font-weight:900;letter-spacing:8px;color:#fff}
.btn{display:inline-block;background:#fff;color:#000;padding:14px 32px;border-radius:4px;text-decoration:none;font-weight:700;font-size:14px;letter-spacing:1px;text-transform:uppercase;margin:20px 0}
.footer{background:#000;padding:20px;text-align:center;font-size:12px;color:#666;border-top:1px solid #222}
</style></head>
<body>
<div class='container'>
  <div class='header'><h1>ProCRM</h1></div>
  <div class='body'>$content</div>
  <div class='footer'>&copy; " . date('Y') . " ProCRM. All rights reserved.</div>
</div>
</body></html>";
}

// ─── Settings ────────────────────────────────────────────────────────────────
function getSettings() {
    $db = getDB();
    $result = $db->query("SELECT setting_key, setting_value FROM settings");
    $settings = [];
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
    return $settings;
}

function getSetting($key, $default = '') {
    $db = getDB();
    $stmt = $db->prepare("SELECT setting_value FROM settings WHERE setting_key=?");
    $stmt->bind_param('s', $key);
    $stmt->execute();
    $res = $stmt->get_result()->fetch_assoc();
    return $res ? $res['setting_value'] : $default;
}

// ─── Flash Messages ───────────────────────────────────────────────────────────
function setFlash($type, $message) {
    $_SESSION['flash'] = ['type' => $type, 'message' => $message];
}

function getFlash() {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

function renderFlash() {
    $flash = getFlash();
    if (!$flash) return '';
    $icon = $flash['type'] === 'success' ? '✓' : ($flash['type'] === 'error' ? '✕' : 'ℹ');
    return "<div class='flash flash-{$flash['type']}'><span class='flash-icon'>$icon</span> " . htmlspecialchars($flash['message']) . "</div>";
}

// ─── Upload ───────────────────────────────────────────────────────────────────
function uploadFile($file, $dir = 'uploads/payments') {
    $allowed = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'application/pdf'];
    if (!in_array($file['type'], $allowed)) return false;
    if ($file['size'] > 10 * 1024 * 1024) return false;
    
    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid('pay_', true) . '.' . $ext;
    $fullDir = __DIR__ . '/../' . $dir;
    if (!is_dir($fullDir)) mkdir($fullDir, 0755, true);
    
    if (move_uploaded_file($file['tmp_name'], $fullDir . '/' . $filename)) {
        return $dir . '/' . $filename;
    }
    return false;
}
